//////////////////////////////////////////////////////////////////////////////
//  Copyright 2014 by Autodesk, Inc.  All rights reserved.
//
// This computer source code and related instructions and comments 
// are the unpublished confidential and proprietary information of 
// Autodesk, Inc. and are protected under applicable copyright and 
// trade secret law.  They may not be disclosed to, copied or used 
// by any third party without the prior written consent of Autodesk, Inc.
//////////////////////////////////////////////////////////////////////////////

/*globals $,jQuery,document, window, navigator*/

/*!
 * ntp.page - Implements animations and functionality of new tab page elements
 */

 // Namespace
var ntp = ntp || {};
ntp.page = ntp.page || {};

ntp.page.animationSettings = {
    TAB_SWITCHING_ANIMATION_DURATION: 600,
    ELEMENT_TOGGLING_ANIMATION_DURATION: 200
};

/**
 * Handles page layout that cannot be done via CSS
 */
ntp.page.layout = (function () {
    var $allTabs = null;
    var $columns = null;
    var $getStartedVideo = null;
    var $featuredVideo = null;
    var $onlineResource = null;
    var $continueWorking = null;
    var $connectPanel = null;
    var columnMinHeight = 504;
    //This is video item height. thumb height 90px, thumb border is 2px and thumb bottom-margin is 8px.
    var videoItemHeight = 100;
    //This is video item margin. video item margin-bottom is 16px.
    var videoItemGap = 16;
    
   
    var getColumnHeight = function() {
        var newHeight = $allTabs.height() * 0.7;
        if (newHeight < columnMinHeight){
            newHeight = columnMinHeight;
        }
        return newHeight;
    };
    var adjustContentSpace = function () {
        var allTabsHeight = $allTabs.height();
        var newColumnHeight = allTabsHeight * 0.7;
        var space = (allTabsHeight - newColumnHeight) * 0.375;
        if (space < 0) {
            space = 0;
        }
        // adjust top space
        $allTabs.children('.content').css('margin-top', space);
        // adjust content height
        $allTabs.children('.content').css('height', allTabsHeight - space);
    };
    
    return {
        init: function () {
            $canvas = $('#canvas');
            $allTabs = $('#canvas_slider > .tab');
            $columns = $('section.column');
            $getStartedVideo = $('#getting_started ol');
            $featuredVideo = $('#feature_videos ol');
            $continueWorking = $('#continue_working');
            $onlineResource = $('#online_resources_panel');
            $connectPanel = $('#connect_scroll_panel');
            this.adjustColumnHeight();
        },
        
        adjustColumnHeight: function () {
            var newHeight = getColumnHeight();
            $columns.height(newHeight);
            $continueWorking.height(newHeight);
            this.updateVideoSectionHeight();
            this.updateOnlineResourceHeight();
            this.updateConnectHeight();
        },
        
        update: function () {
            // Update tab width
            var tabWidth = $canvas.width();
            $allTabs.each(function () {
                $(this).width(tabWidth);
            });
            
            this.adjustColumnHeight();
            adjustContentSpace();
        },
        
        updateVideoSectionHeight: function() {
            var height = getColumnHeight();
            if(ntp.content.learnTab.hasGetStartedVideos() && ntp.content.learnTab.hasFeaturedVideos()) {
                var getStartedItemsHeight = $getStartedVideo.children().length * videoItemHeight
                                                + ($getStartedVideo.children().length - 1) * videoItemGap;
                var featuredItemsHeight = $featuredVideo.children().length * videoItemHeight
                                                + ($featuredVideo.children().length - 1) * videoItemGap;
                var getStartedHeight = height * 0.5;
                if (getStartedHeight > getStartedItemsHeight)
                    getStartedHeight = getStartedItemsHeight;
                var featuredHeight = height - getStartedHeight
                                        - $('#getting_started_title').outerHeight(true)
                                        - $('#video_section_spacing').outerHeight(true)
                                        - $('#featured_title').outerHeight(true);
                if (featuredHeight > featuredItemsHeight){
                    featuredHeight = featuredItemsHeight;
                    getStartedHeight = height - featuredHeight
                                        - $('#getting_started_title').outerHeight(true)
                                        - $('#video_section_spacing').outerHeight(true)
                                        - $('#featured_title').outerHeight(true);
                }
                $getStartedVideo.css({'height': 'auto', 'max-height': getStartedHeight});
                $featuredVideo.css({'height': 'auto','max-height': featuredHeight});
            }
            else if(ntp.content.learnTab.hasGetStartedVideos()) {
                var getStartedHeight = height - $('#getting_started_title').outerHeight(true);
                $getStartedVideo.css({'height': 'auto','max-height': getStartedHeight});
            }
            else if(ntp.content.learnTab.hasFeaturedVideos()) {
                var featuredHeight = height - $('#featured_title').outerHeight(true);
                $featuredVideo.css({'height': 'auto', 'max-height': featuredHeight});
            }
        },
        
        updateOnlineResourceHeight: function() {
            if(ntp.content.learnTab.hasOnlineResource()) {
                var height = getColumnHeight();
                var tipHeight = 0;
                if(ntp.content.learnTab.hasTips()) {
                    tipHeight = $('#Section_tip').outerHeight(true);
                }
                var onlineHeight = height - $('#online_resource_title').outerHeight(true) - tipHeight;
                $onlineResource.css('max-height', onlineHeight);
            }
        },
        
        updateConnectHeight: function() {
            if(ntp.widget.connect.hasConnect()){
                var height = getColumnHeight();
                var panelHeight;
                if(ntp.widget.notifications.hasNotifications()){
                    panelHeight = height - $('#connect_title').outerHeight(true) - $('#section_notifications').outerHeight(true);
                }
                else {
                    panelHeight = height - $('#connect_title').outerHeight(true);
                }
                $connectPanel.css('max-height', panelHeight);
            }
        }
    };
})();

/**
 * Handles tabs and tab switching
 */
ntp.page.tabs = (function () {
    
    var $canvas = null;
    var $tabs = null;
    var tipContentsVisited = false;
    var $visableTabs = null;
    var selectedIndex = 0;
    
    return {
        current: function () {
            return selectedIndex;
        },

        isTipContentsVisited: function () {
            return tipContentsVisited;
        },
        
        count: function () {
            if ($tabs) {
                return $tabs.length;
            } else {
                return 0;
            }
        },
        
        visibleCount: function () {
            if ($visibleTabs)
                return $visibleTabs.length;
            else
                return 1;
        },
        
        init: function () {
            $canvas = $('#canvas');
            $tabs = $('#canvas_slider > .tab');
            $visibleTabs = $('#canvas_slider > .tab.show');
        },
        
        addVisibleTab: function (id, select) {
            var index = 1;
            var $tab = $visibleTabs.eq(selectedIndex - 1);
            var overCurrent = false;
            for(var i = 0; i < $tabs.length; i++){
                if($tabs.eq(i).attr('id') === id){
                    $tabs.eq(i).addClass('show');
                    $('#footer_nav li').eq(i).addClass('show');
                    $visibleTabs = $('#canvas_slider > .tab.show');
                    if(!overCurrent){
                        selectedIndex++;
                        if(!select)
                            this.update();
                    }
                    ntp.page.footerNavs.visibleTabsChange();
                    ntp.page.sideNavs.visibleTabsChange();
                    break;
                }
                else if($tabs.eq(i).attr('id') === $tab.attr('id')){
                    overCurrent = true;
                }
            }
            if(select)
                this.selectById(id, true, true);
        },
        
        removeVisibleTab: function (id, defaultId) {
            $visibleTabs = $('#canvas_slider > .tab.show');
            var $tab = $visibleTabs.eq(selectedIndex - 1);
            if ($tab.attr('id') === id) {
                $tab.removeClass('show');
                $('#footer_nav li').eq(selectedIndex - 1).removeClass('show');
                $visibleTabs = $('#canvas_slider > .tab.show');
                this.selectById(defaultId, true, true);
                ntp.page.footerNavs.visibleTabsChange();
                ntp.page.sideNavs.visibleTabsChange();
            }
            else {
                var overCurrent = false;
                for (var i = 0; i < $visibleTabs.length; i++) {
                    if ($visibleTabs.eq(i).attr('id') === id) {
                        $visibleTabs.eq(i).removeClass('show');
                        $('#footer_nav li').eq(i).removeClass('show');
                        $visibleTabs = $('#canvas_slider > .tab.show');
                        if (!overCurrent) {
                            selectedIndex--;
                            this.update();
                        }
                        ntp.page.footerNavs.visibleTabsChange();
                        ntp.page.sideNavs.visibleTabsChange();
                        break;
                    } else if ($visibleTabs.eq(i).attr('id') === $tab.attr('id')) {
                        overCurrent = true;
                    }
                }
            }
        },
        
        // Selects a tab by its index that is only count for visible tabs.
        selectByIndex: function(index, immediate, forced) {
            var indexChanged = true;
            if (selectedIndex === index) {
                if (!forced) { return; }
                indexChanged = false;
            }
            
            selectedIndex = index;

            // Updates footer navigator class
            $('#footer_nav li.show').each(function (index) {
                if (index + 1 === selectedIndex) {
                    $(this).addClass('highlight');
                } else {
                    $(this).removeClass('highlight');
                }
            });

            // If 'immediate' is auto, then an animation will be used only if canvas is already running an animation.
            if (immediate === 'auto') {
                immediate = !$canvas.is(':animated');
            }

            // Updates tab position
            var $tab = $('.tab.show').eq(index - 1);

            if (tipContentsVisited === true && ntp.content.learnTab.urlIsHilightedContent('tip')) {
                //Leaving learn tab, so unhilight tip
                ntp.content.learnTab.unhilightViewedLearnConents({ tip: $('div#tip_resource_box').data('upload_time') });
                tipContentsVisited = false;

            }else if ($tab.attr('id') === 'tab_learn' && ntp.content.learnTab.urlIsHilightedContent('tip')) {

                //First time Visiting, so notify all others except self.
                ntp.acad.updateLearnContentState('tip', $('div#tip_resource_box').data('upload_time'));
                ntp.acad.updateLearnContentState('tipID', $('div#tip_resource_box').data('tip_ID'));
                ntp.acad.updateLearnContentState('tipExpireDate', $('div#tip_resource_box').data('tip_Expire_Date'));
                tipContentsVisited = true;
            }

            var newScrollLeft = $tab.position().left;
            if ($canvas.scrollLeft() !== newScrollLeft) {
                if (immediate) {
                    $canvas.stop().scrollLeft(newScrollLeft);
                } else {
                    var durationMultiplier = Math.abs(newScrollLeft - $canvas.scrollLeft()) / $canvas.width();
                    if (durationMultiplier > 1) { durationMultiplier = 1; }
                    $canvas.stop().animate(
                        { scrollLeft: newScrollLeft },
                        durationMultiplier * ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION,
                        'linear');
                }
            }
            ntp.page.footerNavs.updateArrow(immediate);
            ntp.page.sideNavs.update(immediate, true, true);
        },
        
        update: function () {
            this.selectByIndex(selectedIndex, 'auto', true);
        },

        // Selects a tab by its id
        selectById: function(id, immediate, forced) {
            var tabIndex = $('.tab.show').index($('#' + id));
            if (tabIndex >= 0 && tabIndex < this.visibleCount()) {
                this.selectByIndex(tabIndex + 1, immediate, forced);
            }
        },
        
        // Auto snaps to the closest tab after scrolling
        snap: function (immediate) {
            var tabWidth = $canvas.width();
            var currentScrollLeft = $canvas.scrollLeft();
            
            // Return if scrolling is caused by resizing
            if (currentScrollLeft === (tabWidth * (selectedIndex - 1))) {
                return;
            }
            
            var closestTabIndex = Math.floor((currentScrollLeft + tabWidth / 2) / tabWidth) + 1;
            this.selectByIndex(closestTabIndex, immediate, true);
        }
    };
})();

/**
 * Handles side navigation arrows
 */
ntp.page.sideNavs = (function () {
    
    var $canvas = null;
    var $tabContents = null;
    var $sideNavs = null;
    
    var tabContentWidth = null;
    var leftSideNavWidth = null;
    var rightSideNavWidth = null;
    
    var overlapped = null;
    
    // Updates the global variable storing left and right of the content of currently selected tab        
    var updateCurrentTabContentWidth = function () {
        if ($tabContents === null) { return null; }
        var selectedIndex = ntp.page.tabs.current();
        var $content = $tabContents.eq(selectedIndex - 1);
        var $columns = $content.children('.column');
        if ($columns.length === 0) {
            tabContentWidth = $content.outerWidth(true);
        } else {
            var $firstColumn = $columns.eq(0);
            var $lastColumn = $columns.eq($columns.length - 1);
            tabContentWidth = $lastColumn.offset().left + $lastColumn.outerWidth(true) - $firstColumn.offset().left;
        }
    };
    
    // Updates side navigator overlap status. If side navigators overlap with page content, they would be hidden.
    var updateOverlap = function (immediate) {
        // Update stored tab content width
        updateCurrentTabContentWidth();

        var tabWidth = $canvas.width();
        var sideNavSpace = (tabWidth - tabContentWidth) / 2;
    
        var newOverlapped = ((sideNavSpace - leftSideNavWidth) < 10 || 
                (sideNavSpace -rightSideNavWidth) < 10);

        if (newOverlapped !== overlapped) {
            if (overlapped === null || immediate) {
                if (newOverlapped) {
                    $sideNavs.addClass('overlapped');
                } else {
                    $sideNavs.removeClass('overlapped');
                }
            } else {
                // Use a fadein or fadeout animation when showing or hiding the side navigators
                $sideNavs.stop(true);
                if (newOverlapped) {
                    $sideNavs.css('opacity', 1);
                    $sideNavs.animate({ opacity: 0 }, 
                        ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                        'linear', function () {
                        $sideNavs.css('opacity', '');
                        $sideNavs.addClass('overlapped');
                    });
                } else {
                    $sideNavs.css('opacity', 0);
                    $sideNavs.removeClass('overlapped');
                    $sideNavs.animate({ opacity: 1 }, 
                        ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                        'linear', function () {
                        $sideNavs.css('opacity', '');
                    });
                }
            }

            overlapped = newOverlapped;
        }
    };
    
    // Updates total width of side navigation tools in global variable.
    var updateWidth = function () {
        leftSideNavWidth = 33;
        leftSideNavWidth += $sideNavs.eq(0).outerWidth(true);
        rightSideNavWidth = 33;
        rightSideNavWidth += $sideNavs.eq(1).outerWidth(true);    
    };
    
    // Updates individual side navigator according to currently selected tab.
    var updateNavigator = function (navigatorSelector, navigatorTabIndex, immediate, handleOverlap) {
        var $sideNav = $(navigatorSelector);

        $sideNav.off('click'); // Clear existing event handler

        if (immediate) {
            var sideNavShouldShow = (navigatorTabIndex > 0 && navigatorTabIndex <= ntp.page.tabs.visibleCount()
                                    && ntp.page.tabs.visibleCount() > 1);
            if (sideNavShouldShow) {
                // Target tab should show. 
                var $footerNav = $('#footer_nav li.show').eq(navigatorTabIndex - 1);
                var label = $footerNav.children('.label').text();
                $sideNav.removeClass('disabled')
                        .children('.label')
                        .text(label);

                $sideNav.one('click', function (event) {
                    ntp.page.tabs.selectByIndex(navigatorTabIndex);
                    event.preventDefault();
                });
            } else {
                //Side navigator need to hide.
                $sideNav.addClass('disabled');
            }

            // Update width of side navigators after their size has changed
            updateWidth();

            if (handleOverlap) {
                updateOverlap(true);
            }
        } else {
            // Change navigators with fade in/fade out effect. The implementation is a little bit complicated,
            // as the simple jQuery fadeIn() fadeOut() do not work so well here.
            var fadeIn = function () {
                $sideNav.stop().css('opacity', '');
                updateNavigator(navigatorSelector, navigatorTabIndex, true, handleOverlap);
                if ($sideNav.css('display') !== 'none') {
                    $sideNav.animate({ opacity: 1 }, 
                        ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                        'linear', function () {
                        $sideNav.css('opacity', '');
                    });
                }
            };

            $sideNav.stop();
            if ($sideNav.css('display') === 'none') {
                setTimeout(fadeIn, ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2);
            } else {
                $sideNav.animate({ opacity: 0 }, 
                    ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION / 2, 
                    'linear', fadeIn);
            }
        }
    };
    
    return {
        init: function () {
            $canvas = $('#canvas');         
            $tabContents = $('#canvas_slider > .tab.show > .content');
            $sideNavs = $('#side_nav li');
        },
        
        visibleTabsChange: function () {
            $tabContents = $('#canvas_slider > .tab.show > .content');
            this.update(true, true, true);
        },
        
        // Updates both side navigators according to currently selected tab.
        update: function (immediate, handleOverlap, forced) {
            var currentTab = ntp.page.tabs.current();
            
            if (forced) {
                // Update side navigators
                updateNavigator('#left_side_nav', currentTab - 1, immediate, handleOverlap);
                updateNavigator('#right_side_nav', currentTab + 1, immediate, handleOverlap);
            } else {
                updateOverlap();
            }
        }
    };
})();

/**
 * Handles footer navigation buttons
 */
ntp.page.footerNavs = (function () {
    var $canvas = null;
    var $footerNavs = null;
    var $navArrow = null;
    var $navRoot = null;
    var footerNavWidth = null;
    var isInit = false;
    
    return {
        init: function () {
            $canvas = $('#canvas');
            $navArrow = $('#nav_arrow');
            $footerNavs = $('#footer_nav li.show');
            $navRoot = $('#footer_nav');
            if(!isInit){
                $('#footer_nav li .button').each(
                        function(index) {
                            var tabId = $('#canvas_slider > .tab').eq(index).attr('id');
                            $(this).on("click", function() {
                                ntp.page.tabs.selectById(tabId);
                                event.preventDefault();
                            });
                        });
            }
            isInit = true;
            if($footerNavs.length > 1)
                this.visibleTabsChange();
        },
        
        visibleTabsChange: function () {
            $footerNavs = $('#footer_nav li.show');
            // Makes the footer navs equal width, and store footerNavWidth
            footerNavWidth = 0;

            if ($footerNavs.length > 1) {
                var innerWidth = 0;
                $navRoot.removeClass('hidden');
                $footerNavs.each(function() {
                    $(this).width('auto');
                    var currentNavWidth = $(this).width();
                    if (currentNavWidth > innerWidth) {
                        innerWidth = currentNavWidth;
                    }
                });

                $footerNavs.each(function(index) {
                    if (index + 1 === ntp.page.tabs.current()) {
                        $(this).addClass('highlight');
                    } else {
                        $(this).removeClass('highlight');
                    }
                    $(this).width(innerWidth);
                });

                // Calculate width from offset of two elements. This is needed to count in a mysterious space between the elements in the width
                // (elements already have margin = 0 )
                var firstTabLeft = $footerNavs.eq(0).offset().left;
                var secondTabLeft = $footerNavs.eq(1).offset().left;
                if (firstTabLeft === secondTabLeft && firstTabLeft > 0)
                    footerNavWidth = $footerNavs.eq(0).innerWidth();
                else
                    footerNavWidth = $footerNavs.eq(1).offset().left - $footerNavs.eq(0).offset().left;
                $navArrow.removeClass("hidden");
                this.updateArrow(true);
            } else {
                $navRoot.addClass('hidden');
                $navArrow.addClass("hidden");
            }
        },
        
        updateArrow: function(immediate){
            if(!$('#nav_arrow').hasClass("hidden")){
                var index = ntp.page.tabs.current() - 1;
                var footerFirstTab = $footerNavs.eq(0);
                var footerLeft = footerFirstTab.offset().left;
                var newOffset = footerLeft + footerNavWidth * (ntp.page.tabs.current() - 0.5) - $navArrow.width() / 2;
                if(immediate){
                    $navArrow.stop().css('left', newOffset);
                } else {
                    var oldLeft = $navArrow.css('left');
                    if(oldLeft !== newOffset){
                        $navArrow.stop().animate(
                                {left: newOffset},
                                ntp.page.animationSettings.TAB_SWITCHING_ANIMATION_DURATION,
                                'linear');
                    }
                }
            }
        },
        
        update: function () {
            if (!footerNavWidth) { // footerNavWidth might be 0 if the window size is too small
                this.visibleTabsChange();
            }
            this.updateArrow(true);
        }
    };
})();


/**
 * Page initialization on customization complete
 */
ntp.deferred.customization.done(function () {
    ntp.page.layout.init();
    ntp.page.tabs.init();
    ntp.widget.templatebar.init();
    
    if (ntp.page.tabs.count() > 1) {
        ntp.page.sideNavs.init();
    }

    $(window).resize(function() {
        ntp.widget.templatebar.update();
        ntp.widget.projectbar.update();
        ntp.page.layout.update();

        if (ntp.page.tabs.count() > 1) {
            ntp.page.footerNavs.update();
            // Fix the problem of window resizing when switch tab transition is not finished.
            ntp.page.tabs.update();
        }
    });

    $(document).on('keydown', function(e) {
        if (e.ctrlKey || e.altKey) {
            ntp.acad.notifyShortcutKeyPressed(e);
        }
    });
    
    $(document).mousedown(function(e) {
        if(e.button == 1) {
            e.preventDefault();
        }
    });

    // Initialize layouts
    ntp.page.layout.update();
    
    // Show create tab on start
    if (ntp.page.tabs.count() > 1) {
        ntp.page.tabs.selectById('tab_create', true, true);
    }
});

/**
 * Page initialization on customization and localization complete.
 */
$.when(ntp.deferred.localization, ntp.deferred.customization).done(function () {
    // Updates layout after text is loaded. Some elements are resized automatically.
    if (ntp.page.tabs.count() > 1) {
        ntp.page.footerNavs.init();
        ntp.page.sideNavs.update(true, true, true); // Update side navigator names
    }
});

/**
 * Allows only dropping dwg files
 */
(function () {
    function isDrawingFile(path) {
        path = path || "";
        path = path.substr(path.length - 4).toLowerCase();
        return (path == ".dxf") || (path == ".dwg") || (path == ".dws") || (path == ".dwt");
    }

    function queryDropHandler(e) {
        e.preventDefault();
        e.stopPropagation();

        var data = e.originalEvent.dataTransfer;
        data.dropEffect = "none";
        if (data && data.files) {
            for (var i = 0; i < data.files.length; ++i) {
                if (isDrawingFile(data.files[i].name)) {
                    data.dropEffect = "copy";
                    break;
                }
            }
        }
    }

    $(window).on('dragenter', queryDropHandler);
    $(window).on('dragover', queryDropHandler);
    $(window).on('drop', function(e) {
        e.preventDefault();
        e.stopPropagation();
        ntp.acad.openDroppedFiles();
    });
})();

// SIG // Begin signature block
// SIG // MIIZ4gYJKoZIhvcNAQcCoIIZ0zCCGc8CAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // iDuDUVnymFr7LBAPT1BhZBRsfPVKmmTe8w5uLrwOwI6g
// SIG // ggpSMIIFAzCCA+ugAwIBAgIQdnK0JhTS6n5kEFOlgziE
// SIG // ljANBgkqhkiG9w0BAQsFADCBhDELMAkGA1UEBhMCVVMx
// SIG // HTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8w
// SIG // HQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3JrMTUw
// SIG // MwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
// SIG // b2RlIFNpZ25pbmcgQ0EgLSBHMjAeFw0xNjA4MDgwMDAw
// SIG // MDBaFw0xNzA5MDIyMzU5NTlaMIGIMQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECAwKQ2FsaWZvcm5pYTETMBEGA1UEBwwK
// SIG // U2FuIFJhZmFlbDEWMBQGA1UECgwNQXV0b2Rlc2ssIElu
// SIG // YzEfMB0GA1UECwwWRGVzaWduIFNvbHV0aW9ucyBHcm91
// SIG // cDEWMBQGA1UEAwwNQXV0b2Rlc2ssIEluYzCCASIwDQYJ
// SIG // KoZIhvcNAQEBBQADggEPADCCAQoCggEBAOqZ9OjPzARq
// SIG // d9dP4fq2qXdpTJolFYJ8Wxg8WcZVciX3l+dljew4R9C8
// SIG // WR/0tW17lLANaSORF4IYmWTgdpFg5PGR3s+R4A461hpR
// SIG // IG9QthGUq8Bt7af5VbCjjKtYBNq/x3ukqVsw/1/qljGV
// SIG // plDOyr976ktnCm0/wL0N+ubk5WAIgv+I9E0i6+GX1hRm
// SIG // +eRliYKeKF0/gBLyDh3ut+N3HNPrnfjfec7q2Czd82Ce
// SIG // RBCToR1E7Cp6yBBdGUfb3JE0PnE1+6n0qJnsvs2bsCIz
// SIG // qVo3A0iMHBsbioxa5DYyXuEfNCktosUlFXfegd514ZJn
// SIG // s6YDr4PlahC0+lK3hgJy5vkCAwEAAaOCAWkwggFlMAkG
// SIG // A1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQM
// SIG // MAoGCCsGAQUFBwMDMGEGA1UdIARaMFgwVgYGZ4EMAQQB
// SIG // MEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5zeW1jYi5j
// SIG // b20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5z
// SIG // eW1jYi5jb20vcnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL
// SIG // 3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIwIKAeoByGGmh0
// SIG // dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUF
// SIG // BwEBBEswSTAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5
// SIG // bWNkLmNvbTAmBggrBgEFBQcwAoYaaHR0cDovL3JiLnN5
// SIG // bWNiLmNvbS9yYi5jcnQwEQYJYIZIAYb4QgEBBAQDAgQQ
// SIG // MBYGCisGAQQBgjcCARsECDAGAQEAAQH/MA0GCSqGSIb3
// SIG // DQEBCwUAA4IBAQDAO29k596Wq5ympOToYLmaRc7ZeGvY
// SIG // x6j5lNWmwCC9ACLTN+mqXF1msf5KtHRHTpyLFVDH/zCC
// SIG // 3LrRzHSHuflhkKFlhJQsEttZ3rhmKxlEtJ85Id9pA8wm
// SIG // XN+Q5tKIStReWLsZ2eA15G9BEFmPvq5DAFj0h+LNF6hq
// SIG // 87C2bVqim29Kf9wDMp3Ndd7hj07QEVh7CqCKEoAJYvXR
// SIG // BBAdrMTqjCTtYCwDkAaAg5LdEm5w76jCQkR2XzKOTgDl
// SIG // qa9uQIyQBAc2ci6X9OBdGJw0ZM0JDlEkRpr3uODtZIOn
// SIG // UQQchI+k+cwEAK7+vFhfsLBi7Dyt1mbvfINr803cFja3
// SIG // i5JOMIIFRzCCBC+gAwIBAgIQfBs1NUrn23TnQV8Racpr
// SIG // qDANBgkqhkiG9w0BAQsFADCBvTELMAkGA1UEBhMCVVMx
// SIG // FzAVBgNVBAoTDlZlcmlTaWduLCBJbmMuMR8wHQYDVQQL
// SIG // ExZWZXJpU2lnbiBUcnVzdCBOZXR3b3JrMTowOAYDVQQL
// SIG // EzEoYykgMjAwOCBWZXJpU2lnbiwgSW5jLiAtIEZvciBh
// SIG // dXRob3JpemVkIHVzZSBvbmx5MTgwNgYDVQQDEy9WZXJp
// SIG // U2lnbiBVbml2ZXJzYWwgUm9vdCBDZXJ0aWZpY2F0aW9u
// SIG // IEF1dGhvcml0eTAeFw0xNDA3MjIwMDAwMDBaFw0yNDA3
// SIG // MjEyMzU5NTlaMIGEMQswCQYDVQQGEwJVUzEdMBsGA1UE
// SIG // ChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xHzAdBgNVBAsT
// SIG // FlN5bWFudGVjIFRydXN0IE5ldHdvcmsxNTAzBgNVBAMT
// SIG // LFN5bWFudGVjIENsYXNzIDMgU0hBMjU2IENvZGUgU2ln
// SIG // bmluZyBDQSAtIEcyMIIBIjANBgkqhkiG9w0BAQEFAAOC
// SIG // AQ8AMIIBCgKCAQEA15VD1NzfZ645+1KktiYxBHDpt45b
// SIG // Kro3aTWVj7vAMOeG2HO73+vRdj+KVo7rLUvwVxhOsY2l
// SIG // M9MLdSPVankn3aPT9w6HZbXerRzx9TW0IlGvIqHBXUuQ
// SIG // f8BZTqudeakC1x5JsTtNh/7CeKu/71KunK8I2TnlmlE+
// SIG // aV8wEE5xY2xY4fAgMxsPdL5byxLh24zEgJRyu/ZFmp7B
// SIG // JQv7oxye2KYJcHHswEdMj33D3hnOPu4Eco4X0//wsgUy
// SIG // GUzTsByf/qV4IEJwQbAmjG8AyDoAEUF6QbCnipEEoJl4
// SIG // 9He082Aq5mxQBLcUYP8NUfSoi4T+IdpcXn31KXlPsER0
// SIG // b21y/wIDAQABo4IBeDCCAXQwLgYIKwYBBQUHAQEEIjAg
// SIG // MB4GCCsGAQUFBzABhhJodHRwOi8vcy5zeW1jZC5jb20w
// SIG // EgYDVR0TAQH/BAgwBgEB/wIBADBmBgNVHSAEXzBdMFsG
// SIG // C2CGSAGG+EUBBxcDMEwwIwYIKwYBBQUHAgEWF2h0dHBz
// SIG // Oi8vZC5zeW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBka
// SIG // F2h0dHBzOi8vZC5zeW1jYi5jb20vcnBhMDYGA1UdHwQv
// SIG // MC0wK6ApoCeGJWh0dHA6Ly9zLnN5bWNiLmNvbS91bml2
// SIG // ZXJzYWwtcm9vdC5jcmwwEwYDVR0lBAwwCgYIKwYBBQUH
// SIG // AwMwDgYDVR0PAQH/BAQDAgEGMCkGA1UdEQQiMCCkHjAc
// SIG // MRowGAYDVQQDExFTeW1hbnRlY1BLSS0xLTcyNDAdBgNV
// SIG // HQ4EFgQU1MAGIknrOUvdk+JcobhHdglyA1gwHwYDVR0j
// SIG // BBgwFoAUtnf6aUhHn1MS1cLqBzJ2B9GXBxkwDQYJKoZI
// SIG // hvcNAQELBQADggEBAH/ryqfqi3ZC6z6OIFQw47e53PpI
// SIG // PhbHD0WVEM0nhqNm8wLtcfiqwlWXkXCD+VJ+Umk8yfHg
// SIG // lEaAGLuh1KRWpvMdAJHVhvNIh+DLxDRoIF60y/kF7Zyv
// SIG // cFMnueg+flGgaXGL3FHtgDolMp9Er25DKNMhdbuX2IuL
// SIG // jP6pBEYEhfcVnEsRjcQsF/7Vbn+a4laS8ZazrS359N/a
// SIG // iZnOsjhEwPdHe8olufoqaDObUHLeqJ/UzSwLNL2LMHhA
// SIG // 4I2OJxuQbxq+CBWBXesv4lHnUR7JeCnnHmW/OO8BSgEJ
// SIG // JA4WxBR5wUE3NNA9kVKUneFo7wjw4mmcZ26QCxqTcdQm
// SIG // AsPAWiMxgg7oMIIO5AIBATCBmTCBhDELMAkGA1UEBhMC
// SIG // VVMxHTAbBgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9u
// SIG // MR8wHQYDVQQLExZTeW1hbnRlYyBUcnVzdCBOZXR3b3Jr
// SIG // MTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1
// SIG // NiBDb2RlIFNpZ25pbmcgQ0EgLSBHMgIQdnK0JhTS6n5k
// SIG // EFOlgziEljANBglghkgBZQMEAgEFAKB8MBAGCisGAQQB
// SIG // gjcCAQwxAjAAMBkGCSqGSIb3DQEJAzEMBgorBgEEAYI3
// SIG // AgEEMBwGCisGAQQBgjcCAQsxDjAMBgorBgEEAYI3AgEV
// SIG // MC8GCSqGSIb3DQEJBDEiBCBA1hIjotmTSK5XS2ljMT1n
// SIG // pUkqOvvO1M/PZvsljaJZ8TANBgkqhkiG9w0BAQEFAASC
// SIG // AQCuv5h+60zj7ULa7rMI7M9/3SHnTLwQ7Ba+Pmik73uQ
// SIG // yP0B8UgFOj46DE9eM4eEgUi1r2Mc/7u1HNFElTbaA3Ty
// SIG // iTOyYIJLnM9WYWL84cUmrwqx8rFXWZqwiaHIP4AIFsGm
// SIG // KFJsesqFk/7FhjCGLwSZQUaww+Gs1gVnC+eKKM/sNGiz
// SIG // qx5ZYaswxYIrFwrvlctlDAGXZwaHcXZRMid6DpOJejIO
// SIG // i0f5pKnZolfYqQjiI8j5WJ9FBeAX/ruSuvf6HEHaqjEW
// SIG // tIPoz8eiJBzEaxb5b0tzJWLBdq7OFIyMwtOlzlpNJZkx
// SIG // kRzZjBCfTfD7zu5PojL2/diRc1ZFL5uDjg9MoYIMoTCC
// SIG // DJ0GCisGAQQBgjcDAwExggyNMIIMiQYJKoZIhvcNAQcC
// SIG // oIIMejCCDHYCAQMxDzANBglghkgBZQMEAgEFADCB3QYL
// SIG // KoZIhvcNAQkQAQSggc0EgcowgccCAQEGCSsGAQQBoDIC
// SIG // AzAxMA0GCWCGSAFlAwQCAQUABCAnTpbRS2dPSoDmVniv
// SIG // lHRO5ghmgUBZDbGMtouxaCz/UwIUEeON4gFnChXkyRM5
// SIG // 6rEgsqD/vBoYDzIwMTcwMjAzMDQ1MDM1WqBdpFswWTEL
// SIG // MAkGA1UEBhMCU0cxHzAdBgNVBAoTFkdNTyBHbG9iYWxT
// SIG // aWduIFB0ZSBMdGQxKTAnBgNVBAMTIEdsb2JhbFNpZ24g
// SIG // VFNBIGZvciBBZHZhbmNlZCAtIEcyoIIIxjCCBKkwggOR
// SIG // oAMCAQICEhEhBvEPzmjwm/rlWxjNjyABdzANBgkqhkiG
// SIG // 9w0BAQsFADBbMQswCQYDVQQGEwJCRTEZMBcGA1UEChMQ
// SIG // R2xvYmFsU2lnbiBudi1zYTExMC8GA1UEAxMoR2xvYmFs
// SIG // U2lnbiBUaW1lc3RhbXBpbmcgQ0EgLSBTSEEyNTYgLSBH
// SIG // MjAeFw0xNjA1MjQwMDAwMDBaFw0yNzA2MjQwMDAwMDBa
// SIG // MFkxCzAJBgNVBAYTAlNHMR8wHQYDVQQKExZHTU8gR2xv
// SIG // YmFsU2lnbiBQdGUgTHRkMSkwJwYDVQQDEyBHbG9iYWxT
// SIG // aWduIFRTQSBmb3IgQWR2YW5jZWQgLSBHMjCCASIwDQYJ
// SIG // KoZIhvcNAQEBBQADggEPADCCAQoCggEBALfHkooo2POR
// SIG // y1ANXespRMGCWaXKZM69g7VR5ZTMboCaF2zc/2LmNkNe
// SIG // AcIMZI3Kd572XXdFuV7IJOtBNxFmN6zIzXSbzLPvTOJ/
// SIG // G85zvsmWnTUefPdU92zsoBLWrpmdY8R4X1mpLiL1wyfY
// SIG // sltFYyeQ/4yxPam08w7A8SBlBomdAxyjsFJBhTTrvMvO
// SIG // VPYS/rMBiUqm+lTFH/vTHMDjv5fjP9Ab+UDHG9XrJnxD
// SIG // MMdw8ngRqoVOpQ4NAEo6EXejyiMBgJ7Ik1ZdRsyK2NKq
// SIG // CoSFsolb1TLOQXsYTlTKq9FSXhLTJJ5W8wyP3b2SjnnV
// SIG // QYnDo6DlkfzHZ52HM85xMnMCAwEAAaOCAWcwggFjMA4G
// SIG // A1UdDwEB/wQEAwIHgDBMBgNVHSAERTBDMEEGCSsGAQQB
// SIG // oDIBHjA0MDIGCCsGAQUFBwIBFiZodHRwczovL3d3dy5n
// SIG // bG9iYWxzaWduLmNvbS9yZXBvc2l0b3J5LzAJBgNVHRME
// SIG // AjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMEYGA1Ud
// SIG // HwQ/MD0wO6A5oDeGNWh0dHA6Ly9jcmwuZ2xvYmFsc2ln
// SIG // bi5jb20vZ3MvZ3N0aW1lc3RhbXBpbmdzaGEyZzIuY3Js
// SIG // MFgGCCsGAQUFBwEBBEwwSjBIBggrBgEFBQcwAoY8aHR0
// SIG // cDovL3NlY3VyZS5nbG9iYWxzaWduLmNvbS9jYWNlcnQv
// SIG // Z3N0aW1lc3RhbXBpbmdzaGEyZzIuY3J0MB0GA1UdDgQW
// SIG // BBQtbm7RjeUDgO7nY+mn2doLPFciPTAfBgNVHSMEGDAW
// SIG // gBSSIadKlV1ksJu0HuYAN0fmnUErTDANBgkqhkiG9w0B
// SIG // AQsFAAOCAQEAV51T5N3upSze5L9igKJhhkqfm50kIzCb
// SIG // jyeHL/oEWc5wiD1GUnfEm0XSj723IRhJ2C6H/5Iud/k/
// SIG // CvmgIVwTT+SEKyiHzFwVuROr4hJVw/hFHkkZzqp1DyHo
// SIG // 71H8NCwLMgUJsuQWaa3ZLn7h/C1IvxrTdDUBOt8wQ3Bn
// SIG // ejjXuhHCVvsxSLpb8SESYuB2iZEfSTjUWE15CYqp2m8C
// SIG // 1q3k2ol9TNmxMHBAattFulN2kNxLQhYhz+TSWJTUVWWb
// SIG // dgOsrhgItoMSjEE+X4BFZMiJ1DMXoaFMvT/Ekv5/hfK+
// SIG // sazX9p7LzhMq7gJDT/z/cDU0ozN8z+INMiQgfAA+ozIR
// SIG // 3jCCBBUwggL9oAMCAQICCwQAAAAAATGJxlAEMA0GCSqG
// SIG // SIb3DQEBCwUAMEwxIDAeBgNVBAsTF0dsb2JhbFNpZ24g
// SIG // Um9vdCBDQSAtIFIzMRMwEQYDVQQKEwpHbG9iYWxTaWdu
// SIG // MRMwEQYDVQQDEwpHbG9iYWxTaWduMB4XDTExMDgwMjEw
// SIG // MDAwMFoXDTI5MDMyOTEwMDAwMFowWzELMAkGA1UEBhMC
// SIG // QkUxGTAXBgNVBAoTEEdsb2JhbFNpZ24gbnYtc2ExMTAv
// SIG // BgNVBAMTKEdsb2JhbFNpZ24gVGltZXN0YW1waW5nIENB
// SIG // IC0gU0hBMjU2IC0gRzIwggEiMA0GCSqGSIb3DQEBAQUA
// SIG // A4IBDwAwggEKAoIBAQCqm47DqxFRJQG2lpTiT9jBCPZG
// SIG // I9lFxZWXW6sav9JsV8kzBh+gD8Y8flNIer+dh56v7sOM
// SIG // R+FC7OPjoUpsDBfEpsG5zVvxHkSJjv4L3iFYE+5NyMVn
// SIG // Cxyys/E0dpGiywdtN8WgRyYCFaSQkal5ntfrV50rfCLY
// SIG // FNfxBx54IjZrd3mvr/l/jk7htQgx/ertS3FijCPxAzmP
// SIG // RHm2dgNXnq0vCEbc0oy89I50zshoaVF2EYsPXSRbGVQ9
// SIG // JsxAjYInG1kgfVn2k4CO+Co4/WugQGUfV3bMW44ETyyo
// SIG // 24RQE0/G3Iu5+N1pTIjrnHswJvx6WLtZvBRykoFXt3bJ
// SIG // 2IAKgG4JAgMBAAGjgegwgeUwDgYDVR0PAQH/BAQDAgEG
// SIG // MBIGA1UdEwEB/wQIMAYBAf8CAQAwHQYDVR0OBBYEFJIh
// SIG // p0qVXWSwm7Qe5gA3R+adQStMMEcGA1UdIARAMD4wPAYE
// SIG // VR0gADA0MDIGCCsGAQUFBwIBFiZodHRwczovL3d3dy5n
// SIG // bG9iYWxzaWduLmNvbS9yZXBvc2l0b3J5LzA2BgNVHR8E
// SIG // LzAtMCugKaAnhiVodHRwOi8vY3JsLmdsb2JhbHNpZ24u
// SIG // bmV0L3Jvb3QtcjMuY3JsMB8GA1UdIwQYMBaAFI/wS3+o
// SIG // LkUkrk1Q+mOai97i3Ru8MA0GCSqGSIb3DQEBCwUAA4IB
// SIG // AQAEVoJKfNDOyb82ZtG+NZ6TbJfoBs4xGFn5bEFfgC7A
// SIG // QiW4GMf81LE3xGigzyhqA3RLY5eFd2E71y/j9b0zopJ9
// SIG // ER+eimzvLLD0Yo02c9EWNvG8Xuy0gJh4/NJ2eejhIZTg
// SIG // H8Si4apn27Occ+VAIs85ztvmd5Wnu7LL9hmGnZ/I1JgF
// SIG // snFvTnWu8T1kajteTkamKl0IkvGj8x10v2INI4xcKjiV
// SIG // 0sDVzc+I2h8otbqBaWQqtaai1XOv3EbbBK6R127FmLrU
// SIG // R8RWdIBHeFiMvu8r/exsv9GU979Q4HvgkP0gGHgYIl0I
// SIG // LowcoJfzHZl9o52R0wZETgRuehwg4zbwtlC5MYICtDCC
// SIG // ArACAQEwcTBbMQswCQYDVQQGEwJCRTEZMBcGA1UEChMQ
// SIG // R2xvYmFsU2lnbiBudi1zYTExMC8GA1UEAxMoR2xvYmFs
// SIG // U2lnbiBUaW1lc3RhbXBpbmcgQ0EgLSBTSEEyNTYgLSBH
// SIG // MgISESEG8Q/OaPCb+uVbGM2PIAF3MA0GCWCGSAFlAwQC
// SIG // AQUAoIIBFDAaBgkqhkiG9w0BCQMxDQYLKoZIhvcNAQkQ
// SIG // AQQwHAYJKoZIhvcNAQkFMQ8XDTE3MDIwMzA0NTAzNVow
// SIG // LwYJKoZIhvcNAQkEMSIEIGynwDq0Wwlb/j1orIz48N44
// SIG // lteplNf+bFG/h1PtxVefMIGmBgsqhkiG9w0BCRACDDGB
// SIG // ljCBkzCBkDCBjQQUfVXY51pWovxzgkP3uFSHXFy1Kg0w
// SIG // dTBfpF0wWzELMAkGA1UEBhMCQkUxGTAXBgNVBAoTEEds
// SIG // b2JhbFNpZ24gbnYtc2ExMTAvBgNVBAMTKEdsb2JhbFNp
// SIG // Z24gVGltZXN0YW1waW5nIENBIC0gU0hBMjU2IC0gRzIC
// SIG // EhEhBvEPzmjwm/rlWxjNjyABdzANBgkqhkiG9w0BAQEF
// SIG // AASCAQB3cUduPWEQ8kDmZCyRikpoFH7Luh7s7y643+RP
// SIG // A72uT6ayWT1UzZP6GMYJcbOZbrvIG821+tqkdLlK15l/
// SIG // mBAfSUfw0mMEJadhSOdCQLNvcZ0IQ+jXpZc/XfisXOq/
// SIG // YhuQK1NNqV8uzbVs21SU5vxXE+kNCgkrbW8GgwHohg7P
// SIG // 3gD5CvLUwbT402v4y3UghGETkg5hGDZTjtHMgKF/+yrG
// SIG // iRSgUzU9o2X6p5A4KByZsgE5Q9f/jKRxnhLAgjLUvMGa
// SIG // PpuMPOcZuISA3pFbC8pLy5FxuyJAvjD9pgl1LXJMnFzF
// SIG // IE3bKMz0/LtIitbdGpuScBckBZOcfL7Q8uiqP0mK
// SIG // End signature block
